/* SCE CONFIDENTIAL
* PlayStation(R)3 Programmer Tool Runtime Library 430.001
* Copyright (C) 2012 Sony Computer Entertainment Inc.
* All Rights Reserved.
*/

#ifndef _SCE_NP_TOOLKIT_DEFINES_H_
#define _SCE_NP_TOOLKIT_DEFINES_H_

#include <sys/sys_time.h>
#include <sys/paths.h>
#include <netex/libnetctl.h>
#include "np.h"
#include "matching_defines.h"
#include <np/commerce2.h>
#include <vector>
#include <list>
#include <string>
#include <cell/sysmodule.h>
#include <cell/http.h>
#include <cell/ssl.h>
#include <np/sns.h>
#include "future.h"

#define SCE_TOOLKIT_NP_MAX_FRIEND_LINE 101	///< The maximum number of friends.
#define SCE_TOOLKIT_NP_MAX_NUM_BOARD	4	///< The maximum number of boards.
#define SCE_TOOLKIT_NP_MAX_RANGE		30	///< The maximum range of ranking scores requested.
#define SCE_TOOLKIT_NP_SKU_PRICE_LEN	(SCE_NP_COMMERCE2_CURRENCY_CODE_LEN \
										+ SCE_NP_COMMERCE2_CURRENCY_SYMBOL_LEN \
										+ SCE_NP_COMMERCE2_THOUSAND_SEPARATOR_LEN \
										+ SCE_NP_COMMERCE2_DECIMAL_LETTER_LEN) ///< The maximum length of a price in characters.

#ifdef SCE_NP_ENTITLEMENT_ID_SIZE
#define SCE_TOOLKIT_NP_COMMERCE_ENTITLEMENT_ID_LEN SCE_NP_ENTITLEMENT_ID_SIZE						///< The size of entitlement ID.
#else 
#define SCE_TOOLKIT_NP_COMMERCE_ENTITLEMENT_ID_LEN 32												///< The size of entitlement ID.
#endif

typedef int32_t SceToolkitNpMessageType;
#define SCE_TOOLKIT_NP_MESSAGE_TYPE_CUSTOM_DATA (1) ///< A custom data message.
#define SCE_TOOLKIT_NP_MESSAGE_TYPE_INVITE	  (2)	///< An invite data message.

namespace sce {
	namespace Toolkit{
		namespace NP{

			/// @brief
			/// Contains relevant friend information.
			///
			/// Contains relevant friend information such as the friends %NP ID.
			typedef struct Friend
			{
				SceNpId npid;							///< The friend's %NP ID.
				SceNpBasicPresenceDetails2 presence;	///< The friend's presence. 
				// More can be added here as needed in future
			} Friend;

			/// @brief
			/// Contains a list of friends. 
			///
			/// Contains information on all the friends of the user. 
			typedef std::vector<Friend> FriendsList;

			///	@brief
			///	Holds the user's status string and data.
			///
			///	Holds the user's status string and data.
			typedef struct PresenceDetails{
				std::string status;								///< The status string of the user.
				uint8_t data[SCE_NP_BASIC_MAX_PRESENCE_SIZE];	///< Optional. The binary data for the user.
				size_t size;									///< The size of the binary data.
			}PresenceDetails;

			///	@brief
			///	Holds the user's score for scoreboard registration.
			///
			///	Holds the user's score for scoreboard registration.
			typedef struct RegisterScore{
				SceNpScoreBoardId boardId;			///< The board ID, which the score is to be registered to.
				SceNpScoreRankNumber tmpRank;		///< The user's temporary rank.
				SceNpScoreValue score;				///< The user's score.
				SceNpScoreComment comment;			///< A comment to be passed along with the score.
				SceNpScoreGameInfo gameInfo;		///< Application specific information to be passed along with the score.
			}RegisterScore;

			/// @brief
			/// Holds the user's registered temporary rank.
			///
			/// Holds the user's registered temporary rank. This structure is used when there are delays in registration, and a temporary rank is passed back from the server to the client.
			typedef struct TempRank{
				SceNpScoreRankNumber tempRank;	///< The temporary rank. 
			}TempRank;

			///	@brief
			///	Holds a range of ranks for displaying purposes.
			///
			///	Holds a range of ranks for displaying purposes. It is passed into <c>sceRanksInformation()</c> to retrieve data on all ranks within a specified range.
			typedef struct RankInformation{
				SceNpScoreRankData rankData[SCE_TOOLKIT_NP_MAX_RANGE];		///< An array of rank data.
				SceNpScoreComment comment[SCE_TOOLKIT_NP_MAX_RANGE];		///< An array of comments relating to the rank data.
				SceNpScoreGameInfo gameInfo[SCE_TOOLKIT_NP_MAX_RANGE];		///< An array of game information relating to the rank data.
				CellRtcTick lastSortDate;									///< The time the server created the ranking data.
				SceNpScoreRankNumber totalRecord;							///< The total number of players registered in the target scoreboard.
				SceNpScoreBoardId boardId;									///< The ID of the ranking board.
				int range;													///< The length of the list.
				int start;													///< The starting index for the list.
				int rankReturned;											///< The rank returned.
			}RankInformation;

			///	@brief
			///	Holds a range of ranks from the user's friend list.
			///
			///	Holds a range of ranks from the user's friend list. It is passed into <c>displayFriendScores()</c> to retrieve data all of a user's friends' scores for a target board ID.
			typedef struct FriendsRankInformation{
				SceNpScorePlayerRankData rankData[SCE_TOOLKIT_NP_MAX_FRIEND_LINE];	///< An array of rank data for a user's friend list.
				SceNpScoreComment comment[SCE_TOOLKIT_NP_MAX_FRIEND_LINE];			///< An array of comments relating to the rank data.
				SceNpScoreGameInfo gameInfo[SCE_TOOLKIT_NP_MAX_FRIEND_LINE];		///< An array of game information relating to the rank data.
				SceNpScoreBoardId boardId;											///< The target board ID.		
				int numFriends;														///< The number of friends in the list.
				system_time_t updateTime;											///< The timestamp.
			}__attribute__((aligned(8))) FriendsRankInformation;

			///	@brief
			///	Holds the user's own ranking or the ranking for a single %NP ID.
			///
			/// Holds the user's own ranking or the ranking for a single %NP ID. It is passed into <c>displayUserRank()</c> to retrieve a single piece of ranking information.
			typedef struct UserRankInformation {
				SceNpScorePlayerRankData rankData;		///< An array of ranks for user's friendlist.
				SceNpScoreComment comment;				///< An array of comments to be retrieved along with the ranks information.
				SceNpScoreGameInfo gameInfo;			///< An array of game information retrieved along with the ranks information.
				SceNpScoreBoardId boardId;				///< The target board ID.						
			}__attribute__((aligned(8))) UserRankInformation;

			/// @brief
			/// Contains basic network information.
			///
			/// Contains basic network information. It includes only the most useful bits of information.
			typedef struct NetStateBasic
			{
				int					connectionStatus;								///< The connection status. This maps to <c>SCE_NET_CTL_STATE_XXX</c>.
				char				ipAddress[CELL_NET_CTL_IPV4_ADDR_STR_LEN];		///< The IP address of the network adapter.
				CellNetCtlNatInfo	natInfo;										///< The NAT type.
			}NetStateBasic;

			/// @brief
			/// Contains detailed network information.
			///
			/// Contains detailed network information. Most of this information would only ever be useful during debugging.
			typedef struct NetStateDetailed
			{
				int					connectionStatus;										///< The connection status. This maps to <c>SCE_NET_CTL_STATE_XXX</c>.  
				CellNetCtlNatInfo	natInfo;												///< The NAT type.
				uint32_t			device;													///< The network device being used.
				CellNetCtlEtherAddr ethernetAddress;										///< The MAC address.
				uint32_t			mtu;													///< MTU.
				uint32_t			link;													///< The link connection state.
				uint32_t			linkType;												///< The link type.	
				CellNetCtlEtherAddr bssid;													///< BSSID.
				CellNetCtlSSID		ssid;													///< SSID.
				uint32_t			wlanSecurity;											///< Specifies whether wireless LAN is encrypted.
				uint32_t			auth8021xType;											///< The setting for 802.1X authorization <c>CELL_NET_CTL_8021X_XXX</c>.
				char				auth8021xAuthName[CELL_NET_CTL_AUTH_NAME_LEN];			///< The user ID to be used for 802.1X authorization.
				uint8_t				rssi;													///< The signal strength.
				uint8_t				channel;												///< The wireless channel used.
				uint32_t			ipConfig;												///< Specifies how the IP address is configured.
				char				dhcpHostname[CELL_NET_CTL_HOSTNAME_LEN];				///< The DHCP hostname.
				char				pppoeAuthName[CELL_NET_CTL_AUTH_NAME_LEN];				///< The PPPOE authentication name.	
				char				ipAddress[CELL_NET_CTL_IPV4_ADDR_STR_LEN];				///< The device's IP address.
				char				netmask[CELL_NET_CTL_IPV4_ADDR_STR_LEN];				///< The device's Net mask.
				char				defaultRoute[CELL_NET_CTL_IPV4_ADDR_STR_LEN];			///< The default route IP address.			
				char				primaryDNS[CELL_NET_CTL_IPV4_ADDR_STR_LEN];				///< The primary domain name server IP address.
				char				secondaryDNS[CELL_NET_CTL_IPV4_ADDR_STR_LEN];			///< The secondary domain name server IP address.
				uint32_t			httpProxyConfig;										///< The configuration of the proxy server.
				char				httpProxyServer[CELL_NET_CTL_HOSTNAME_LEN];				///< The IP address of the proxy.
				uint16_t			httpProxyPort;											///< The proxy server port address.
				uint32_t			uPnpConfig;												///< UPnP setting <c>CELL_NET_CTL_UPNP_XXX</c>.
			}NetStateDetailed;

			/// @brief
			/// Contains the parameters needed to browse a category.
			///
			/// Contains the parameters needed to browse a category.
			typedef struct CategoryBrowseParams
			{
				char	productId[SCE_NP_COMMERCE2_PRODUCT_ID_LEN]; ///< The ID of the product to launch.
				int		userData;									///< Arbitrary data that can be obtained when rebooting game after store browsing.
			}CategoryBrowseParams;

			/// @brief
			/// Contains the parameters needed to browse a product.
			///
			/// Contains the parameters needed to browse a product.
			typedef struct ProductBrowseParams
			{
				char					productId[SCE_NP_COMMERCE2_PRODUCT_ID_LEN];	///< The ID of the product to launch to
				bool					inGame;										///< A  flag that specifies whether to launch as an in-game Store overlay or exit to the game to launch the Store.
				sys_memory_container_t	*memContainer;								///< Only specify a memory container if <c><i>inGame</i></c> is true.
				int						userData;									///< Arbitrary data that can be obtained when rebooting game after store browsing. Only used when <c><i>inGame</i></c> is false.
			}ProductBrowseParams;

			/// @brief
			/// Contains details of how a voucher will be redeemed.
			///
			/// Contains details of how a voucher will be redeemed.
			typedef struct VoucherInputParams 
			{
				bool					inGame;			///< A flag that specifies whether to launch as an in-game Store overlay or exit to the game to launch the Store.
				sys_memory_container_t  *memContainer;	///< Memory container for Store overlay. Only specify if <c><i>inGame</i></c> is true
				int						userData;		///< Arbitrary data that can be obtained when rebooting game after Store browsing. Only used when <c><i>inGame</i></c> is false.
			}VoucherInputParams;

			/// @brief
			/// Contains the details of an entitlement to consume.
			///
			/// Contains the details of an entitlement to consume.
			typedef struct EntitlementToConsume
			{
				SceNpEntitlementId id;	///< The ID of the entitlement to consume.
				uint32_t consumedCount;	///< The amount to consume.
			}EntitlementToConsume;

			/// @brief
			/// Contains information that is used to consume a specified amount of a consumable service entitlement for a given user.
			///
			/// Contains information that is used to consume a specified amount of a consumable service entitlement for a given user.
			typedef struct ConsumeEntitlementInputParams
			{
				char		entitlementId[SCE_TOOLKIT_NP_COMMERCE_ENTITLEMENT_ID_LEN];	///< The service entitlement ID.
				uint32_t	consumedCount;												///< The amount to consume.
				uint32_t	serviceLabel;												///< The PlayStation�4 service label.

				/// @brief
				/// The default constructor.
				///
				/// The default constructor.
				ConsumeEntitlementInputParams() : consumedCount(0), serviceLabel(0)
				{
					memset(entitlementId, 0x0, SCE_TOOLKIT_NP_COMMERCE_ENTITLEMENT_ID_LEN);
				}
			}ConsumeEntitlementInputParams;

			/// @brief
			/// Contains information about a subcategory on the PlayStation(R)Store.
			///
			/// Contains information about a subcategory on the PlayStation(R)Store.
			typedef struct CategoryInfoSub
			{
				char		categoryId[SCE_NP_COMMERCE2_CATEGORY_ID_LEN];					///< The ID of the subcategory.
				char		categoryName[SCE_NP_COMMERCE2_CATEGORY_NAME_LEN];				///< The name of the subcategory.
				char		categoryDescription[SCE_NP_COMMERCE2_CATEGORY_DESCRIPTION_LEN];	///< The detailed description of the subcategory.
				char		imageUrl[SCE_NP_COMMERCE2_URL_LEN];								///< The image URL of the subcategory.
			}CategoryInfoSub;

			/// @brief
			/// Current category information
			///
			/// This structure contains information about a category on the PlayStation(R)Store
			typedef struct CategoryInfo
			{
				CategoryInfoSub				current;				///< The currently selected subcategory.
				std::list<CategoryInfoSub>	subCategories;			///< Information about the subcategories in this category.
				uint32_t					countOfProducts;		///< The number of products in the category.
				uint32_t					countOfSubCategories;	///< The number of subcategories.
			}CategoryInfo;

			/// @brief
			/// Contains information about a product in the PlayStation(R)Store.
			///
			/// Contains information about a product in the PlayStation(R)Store.
			typedef struct ProductInfo
			{
				uint32_t	purchasabilityFlag;													///< A flag that indicates whether the product can be purchased (<c>SCE_NP_COMMERCE2_SKU_PURCHASABILITY_FLAG_XXX</c>).
				char		productId[SCE_NP_COMMERCE2_PRODUCT_ID_LEN];							///< The product ID.
				char		productName[SCE_NP_COMMERCE2_PRODUCT_NAME_LEN];						///< The name of the product.
				char		shortDescription[SCE_NP_COMMERCE2_PRODUCT_SHORT_DESCRIPTION_LEN];	///< A short description of the product.
				char		spName[SCE_NP_COMMERCE2_SP_NAME_LEN];								///< The service provider name.
				char		imageUrl[SCE_NP_COMMERCE2_URL_LEN];									///< The product image URL.
				char		price[SCE_TOOLKIT_NP_SKU_PRICE_LEN];								///< The price of the product. This is formatted to include the currency code.	
				char        padding[6];															///< Padding.	
				CellRtcTick	releaseDate;														///< The product release date.
			}ProductInfo;

			/// @brief
			/// Contains detailed information about a product on the PlayStation(R)Store.
			///
			/// Contains detailed information about a product on the PlayStation(R)Store.
			typedef struct ProductInfoDetailed
			{
				uint32_t	purchasabilityFlag;													///< A flag that indicates whether the product can be purchased (<c>SCE_NP_COMMERCE2_SKU_PURCHASABILITY_FLAG_XXX</c>).
				char		skuId[SCE_NP_COMMERCE2_SKU_ID_LEN];									///< The SKU ID
				char		productId[SCE_NP_COMMERCE2_PRODUCT_ID_LEN];							///< The product ID.
				char		productName[SCE_NP_COMMERCE2_PRODUCT_NAME_LEN];						///< The name of the product.
				char		shortDescription[SCE_NP_COMMERCE2_PRODUCT_SHORT_DESCRIPTION_LEN];	///< A short description of the product.
				char		longDescription[SCE_NP_COMMERCE2_PRODUCT_LONG_DESCRIPTION_LEN];		///< A long description of the product.
				char		legalDescription[SCE_NP_COMMERCE2_PRODUCT_LEGAL_DESCRIPTION_LEN];	///< The legal description for the product.	
				char		spName[SCE_NP_COMMERCE2_SP_NAME_LEN];								///< The service provider name.
				char		imageUrl[SCE_NP_COMMERCE2_URL_LEN];									///< The product image URL.
				char		price[SCE_TOOLKIT_NP_SKU_PRICE_LEN];								///< The price of the product. This is formatted to include the currency code.
				char		ratingSystemId[SCE_NP_COMMERCE2_RATING_SYSTEM_ID_LEN];				///< The ID of the rating system (for example: PEGI, ESRB).
				char		ratingImageUrl[SCE_NP_COMMERCE2_URL_LEN];							///< The URL of the rating icon.
				char		padding[2];															///< Padding.	
				std::list<SceNpCommerce2ContentRatingDescriptor> ratingDescriptors;				///< The list of rating descriptors.			
				CellRtcTick	releaseDate;														///< The product release date.
			}ProductInfoDetailed;

			/// @brief
			/// Contains the input parameters used when retrieving product information.
			///
			/// Contains the input parameters used when retrieving product information.
			typedef struct ProductInfoInputParams 
			{
				char productId[SCE_NP_COMMERCE2_PRODUCT_ID_LEN];	///< The product ID.
				char categoryId[SCE_NP_COMMERCE2_CATEGORY_ID_LEN];	///< The category ID.
			}ProductInfoInputParams;

			/// @brief
			/// Checkout parameters
			///
			/// This structure contains list of SKUs to checkout to and a memory container
			typedef struct CheckoutInputParams 
			{
				std::list<char *>		skuIds;			///< List of SKU IDs
				sys_memory_container_t	*memContainer;	///< Memory container for checkout overlay
			}CheckoutInputParams;

			/// @brief
			/// Contains download list parameters.
			///
			/// Contains download list parameters.
			typedef struct DownloadListInputParams 
			{
				std::list<char *>		skuIds;			///< The list of SKU IDs
				sys_memory_container_t	*memContainer;	///< A memory container for checkout overlay.
			}DownloadListInputParams;

			/// @brief
			/// Contains information that is used to retrieve detailed information for a list of products.
			///
			/// Contains information that is used to retrieve detailed information for a list of products.
			typedef struct DetailedProductInfoListInputParams
			{
				/// @brief
				///	Encapsulates a product ID.
				///
				///	Encapsulates a product ID.
				typedef struct ProductId
				{
					char id[SCE_NP_COMMERCE2_PRODUCT_ID_LEN];	///< The product ID.

					/// @brief
					/// The default constructor.
					///
					/// The default constructor.
					ProductId()
					{
						memset(id, 0x0, SCE_NP_COMMERCE2_PRODUCT_ID_LEN);
					}
				} ProductId;

				std::vector<ProductId>	productIds;						///< The list of product IDs to retrieve the detailed information for. 
				uint32_t				serviceLabel;					///< The PlayStation�4 Service Label.

				/// @brief
				/// The default constructor.
				///
				/// The default constructor.
				DetailedProductInfoListInputParams() : serviceLabel(0)
				{}
			}DetailedProductInfoListInputParams;

			/// @brief
			/// Contains country information.
			///
			/// Contains country information.
			typedef struct CountryInfo
			{
				SceNpCountryCode countryCode;	///< The country code.
				int language;					///< The value stored to language is a CELL_SYSUTIL_LANG_XXX value defined in the system utilities.
			}CountryInfo;

			/// @brief
			/// Contains parental control information.
			///
			/// Contains parental control information.
			typedef struct ParentalControlInfo
			{
				bool isContentRestricted;		///< A flag that specifies whether content is restricted.	
				bool isChatRestricted;			///< A flag that specifies whether chat is restricted.	
				int age;						///< The user's age.
			}ParentalControlInfo;

			/// @brief
			/// Represents a PSN ticket.
			///
			/// Represents a PSN ticket.
			typedef struct Ticket
			{
				void	*buffer;	///< The buffer in which the ticket is stored.
				uint32_t size;		///< The size of the ticket. The maximum file size of the ticket is defined by <c>SCE_NP_TICKET_MAX_SIZE</c>.
				int32_t issuerId;	///< The PSN environment that the user is signed in to.

				/// @brief
				/// The default constructor.
				///
				/// The default constructor
				Ticket(): buffer(0),size(0), issuerId(0)
				{}
			}Ticket;

			/// @brief
			/// Represents a ticket.
			///
			/// Represents a ticket.
			typedef struct AccessCodeInputParams
			{
				std::string clientId;	///< The Auth Client ID. This does not need to be specified if it has already been registered during initialization of the ToolkitNp library.
				std::string scope;		///< The scope of the Access Token required.
			}AccessCodeInputParams;
			
			/// @brief
			/// Contain the components of a message.
			///
			/// Contains the components of a message. 
			typedef struct MessageData { 
				uint8_t				*attachment;				///< The buffer containing the actual data. This should be set using <c>MessageAttachment::setData()</c>.
				size_t				attachmentSize;				///< The number of bytes within the buffer that are being used.
				std::list<SceNpId>	recipients;					///< A list of recipients by identified by <c>SceNpId</c>.
				std::string			subject;					///< The subject string. Must not contain carriage characters.
				std::string			body;						///< The body text of the message.
			}MessageData;

			/// @brief
			/// Contains the components of an invite message.
			///
			/// Contains the components of an invite message.
			typedef struct InviteMessage { 
				std::string body;						///< The body text of the message.
				std::list<SceNpId>	recipients;			///< A list of recipients by identified by <c>SceNpId</c>.
				std::string			subject;			///< The subject string. Must not contain carriage characters.
			}InviteMessage;

			/// @brief
			/// Contains the components of a message.
			///
			/// Contains the components of a message. These
			/// can be left empty, which will cause the user to be prompted to manually
			/// fill in the data using the System Software.
			class  MessageAttachment {
			public:

				/// @brief
				/// The default constructor.
				///
				/// The default constructor. This does not allocate any memory on the heap.
				MessageAttachment();

				/// @brief
				/// The destructor for the MessageAttachment class.
				///
				/// The destructor for the MessageAttachment class. This clear the internal buffer and de-allocates any memory that was allocated to the heap.
				~MessageAttachment();

				/// @brief
				/// Sets the attachment data for the MessageAttachment object.
				///
				/// Sets the attachment data for the MessageAttachment object. Use this method for setting the buffer inside the object. The data given
				/// is copied into an internal buffer which is kept on the heap.
				///
				/// @param data			The data to be copied into the buffer.
				/// @param _size		The size of the data array to be copied in bytes.
				///	
				/// @retval SCE_TOOLKIT_NP_SUCCESS						The operation was successful.
				/// @retval SCE_TOOLKIT_NP_INVALID_ARGUMENT			The operation failed because <c><i>dataSize</i></c> was not a valid integer between 1-<c>SCE_NP_BASIC_MAX_MESSAGE_ATTACHMENT_SIZE</c>.
				/// @retval SCE_TOOLKIT_NP_FAILED_ALLOCATE				The operation failed because the required memory could not be allocated on the heap.
				int			setAttachmentData(uint8_t* data, size_t _size);

				/// @brief
				/// Gets the attachment data contained within the MessageAttachment object.
				///
				/// Gets the attachment data contained within the MessageAttachment object.
				///
				/// @return		A the pointer to the attachment data buffer.
				uint8_t	*getAttachmentData() const {return m_attachment;}

				/// @brief
				/// Gets the size of the attachment data.
				///
				///	Gets the size of the attachment data.
				/// @return		The size of the allocated attachment data buffer.
				size_t		getAttachmentSize() const {return m_attachmentSize;}
			private:
				uint8_t * m_attachment;			///< The buffer containing the actual attachment data. This should be set using setData();
				size_t	  m_attachmentSize;		///< The number of bytes within the buffer that are being used.
			};

			///	@brief
			///	Holds the photo and associated data describing the stream story
			/// to be posted to a Facebook user's wall.
			///
			///	Holds the photo and associated data describing the stream story
			/// to be posted to a Facebook user's wall.
			typedef struct PhotoFb
			{			
				std::string url;			///< A string to hold the URL of the photo, the focal point of the stream story.
				std::string title;			///< A string to hold the accompanying title of the stream story.
				std::string caption;		///< A string to hold the accompanying caption of the stream story.
				std::string description;	///< A string to hold the accompanying description of the stream story.
			}PhotoFb;

			///	@brief
			///	Holds the necessary information needed to describe an action link that
			/// is to be posted to a Facebook user's wall.
			///
			///	Holds the necessary information needed to describe an action link that
			/// is to be posted to a Facebook user's wall. An action link appears beside the "Like"
			/// and "Comment" options on a post.
			typedef struct ActionLinkFb
			{
				std::string name;	///< A string to hold the name of the actionLink that will be displayed as part of the stream story.
				std::string url;	///< A string to hold the actual URL of the page the user will be taken to upon clicking the name.
			}ActionLinkFb;

			///	@brief
			///	Holds the necessary information needed to post a message to a Facebook
			/// user's wall. 
			///
			///	Holds the necessary information needed to post a message to a Facebook
			/// user's wall. It is comprised of the PhotoFb and ActionLinkFb structures, with an
			/// added variable to hold the user's text, that will accompany the post.
			typedef struct SnsPostFacebook
			{
				std::string userText;		///< A string to hold the text that is input by the user, to personalize the stream story.
				PhotoFb photo;				///< The variable to hold the photo information which includes the title, caption and description.
				ActionLinkFb actionLink;	///< The variable to hold the action link information.
			}SnsPostFacebook;

			///	@brief
			///	Holds all the necessary variables for retrieving an access
			/// token from the Facebook Graph API.
			///
			///	Holds all the necessary variables for retrieving an access
			/// token from the Facebook Graph API. It holds additional information such as
			/// permissions that can be specified when requesting a token. The token received
			/// must be encoded before it can be appended to the Facebook Graph API URL.
			typedef struct AccessToken
			{				
				SceNpSnsFbLongAccessTokenResult result;	///< The access token result structure is passed to sceNpSnsFbGetAccessToken() and stores the result.
				std::string	encodedToken;				///< This string stores the encoded result necessary before appending to the Graph API path URL.
				std::string permissions;				///< A string to store permissions for accessing a user's Facebook information.
			}AccessToken;

			///	@brief
			///	Represents the details of a game's trophy set.
			///
			///	Represents the details of a game's trophy set. It contains a <c>SceNpTrophyGameDetails</c> object, which
			///	holds details on the number of trophies a game has, a description of the game and a title's name. It also
			///	contains information on the size of the trophy set icon. 
			typedef struct TrophyGameInfo {
				SceNpTrophyGameDetails gameDetail;	///< Contains information on the trophy set.
				size_t size;						///< The size of the trophy set icon.
				uint8_t* iconData;					///< The trophy set icon in PNG format.
			}TrophyGameInfo;

			///	@brief
			///	Contains information that is used when registering a trophy set through the trophy service.
			///
			///	Contains information that is used when registering a trophy set through the trophy service.
			/// It is required by the <c>Trophy::Interface::registerTrophy</c> function.
			typedef struct TrophyRegister {
				bool cacheTrophyList;		///< A flag that specifies whether to cache the trophy list or not.
				int saveDataInBytes;		///< The minimum amount of free space the title requires for it's save data
				int hddSizeInBytes;			///< The size of free HDD space in bytes.
				TrophyGameInfo gameInfo;	///< Receives the trophy set details.
			}TrophyRegister;

			///	@brief
			///	Contains information that is used when registering a trophy set through the trophy service.
			///
			///	Contains information that is used when registering a trophy set through the trophy service.
			typedef struct TrophyInputMsg {
				bool cacheTrophyList;		///< A flag that specifies whether to cache the trophy list or not.
				bool cacheIcons;			///< A flag that specifies whether to cache the trophy icons.
				int saveDataInBytes;		///< The minimum amount of free space the title requires for its save data.
				int hddSizeInBytes;			///< The size of free HDD space in bytes.
			}TrophyInputMsg;

			///	@brief
			///	Represents information on a trophy such as the ID, grade, name, description etc.
			///
			///	Represents information on a trophy such as the ID, grade, name, description etc.
			///	Also stored is the user's status on each trophy in the trophy set such as whether it is unlocked or not. It is required
			///	by the <c>Trophy::Interface::retrieveTrophyList</c> function.
			typedef struct TrophyInfo {
				SceNpTrophyDetails trophyDetails;	///< This member is a structure which holds static trophy information
				///< such as it's ID, name, description etc.
				SceNpTrophyData trophyData;			///< This member is a structure which holds dynamic trophy information.
				uint8_t* iconData;						///< A buffer which holds trophy icon's data in PNG format.
				size_t size;						///< The icon size of a trophy.
			}TrophyInfo;

			/// @brief
			///	An enumeration which defines the different update types.
			///
			///	An enumeration which defines the different update types.
			///	This enumeration is used in a updateTrophyList function to determine the type of update to carry out.
			///	It is required by the <c>updateTrophyList()</c> function.
			typedef enum UpdateType {
				updateTypeAll=0,		///< This type updates everything in the TrophyInfo structure.
				updateTypeData			///< This type updates only trophyData in the TrophyInfo structure as it contains dynamic information.
				///< Needs updating as user unlocks trophies.
			}TYPE;
		/// @brief
		/// Represents TSS (title small storage) data.
		///
		/// Represents TSS (title small storage) data. This structure contains a pointer to
		/// the start of a buffer and the buffer size. The buffer will be filled with a file, held on a TSS server, which contains the data.
		typedef struct TssData
		{
			void	*buffer;	///< The buffer to store the TSS data in.
			size_t	size;		///< The size of the data buffer. The maximum size of the buffer is defined by #SCE_NET_NP_TSS_MAX_SIZE.
			
			/// @brief
			/// The default constructor.
			///
			/// The default constructor.
			TssData(): buffer(0),size(0) 
			{}
		}TssData;

		/// @brief
		/// Represents a TUS (title user storage) variable.
		///
		/// Represents a TUS (title user storage) variable.
		typedef struct TusVariable
		{
			int32_t		slotId;			///< The slot that the variable belongs to.
			int32_t     reserved;		///< A reserved value.
			int64_t		value;			///< The TUS variable value.

			/// @brief
			/// The default constructor.
			///
			/// The default constructor.
			TusVariable(int32_t s = 0, int64_t v = 0): slotId(s), value(v)
			{}
		}TusVariable;

		/// @brief
		/// Represents TUS (title user storage) data.
		///
		/// Represents TUS (title user storage) data.
		typedef struct TusData
		{
			void	*buffer;	///< The TUS data buffer.
			size_t	bufferSize;	///< The TUS data buffer size.

			/// @brief
			/// The default constructor.
			///
			/// The default constructor.
			TusData(): buffer(0), bufferSize(0)
			{}
		}TusData;

		/// @brief
		/// Represents TUS (title user storage) data output.
		///
		/// Represents TUS (title user storage) data output.
		typedef struct TusDataOutput
		{
			TusData				data;	///< The TUS data.
			SceNpTusDataStatus	status;	///< The status of the data.
		}TusDataOutput;

		/// @brief
		/// Represents the input parameters used when setting TUS (title user storage) variables. 
		///
		/// Represents the input parameters used when setting TUS (title user storage) variables. 
		typedef struct TusSetVarsInputParams
		{
			SceNpId					npid;			///< The NPID of the user to set the variables for.
			std::list<TusVariable>	vars;			///< The list of TUS variables to update.
			bool					isVirtualUser;	///< A flag that specifies whether this update is for a virtual user.  
		}TusSetVarsInputParams;

		/// @brief
		/// Represents the input parameters used when getting TUS (title user storage) variables.
		///
		/// Represents the input parameters used when getting TUS (title user storage) variables.
		typedef struct TusGetVarsInputParams
		{
			SceNpId		npid;			///< The NPID of the user to retrieve the variables for.
			int32_t		*slotIds;		///< An array of integers containing the IDs of the slots to retrieve the variables from.
			size_t		numSlots;		///< The number of slots to retrieve the variables from (the size of the <c><i>slotIds</i></c> array).
			bool		isVirtualUser;	///< A flag that specifies whether it is a virtual user that the data is being retrieved for.
		}TusGetVarsInputParams;

		/// @brief
		/// Represents input parameters used when setting TUS (title user storage) data. 
		///
		/// Represents input parameters used when setting TUS (title user storage) data. 
		typedef struct TusSetDataInputParams
		{
			int32_t		slotId;						///< The ID of the slot that the data belongs to.
			SceNpId		npid;						///< The NPID of the user that is being updated.
			TusData		data;						///< The TUS data to update.
			bool		isVirtualUser;				///< A flag that specifies whether this update is for a virtual user.
			uint64_t	lastChangedDate;			///< Date and time for conflict prevention. Equivalent to SceRtcTick on PS4 and PS Vita, and CellRtcTick on PS3.
													///< Processing is only executed when the time of the last update of the TUS data currently registered on the
													///< server is identical with or older than the specified time. Processing is not performed when no TUS data is
													///< registered on the server. Specify 0 if comparison is not necessary.
			SceNpId		requiredLastChangeUser;		///< NP ID of the author of the update for conflict prevention. Processing is only executed when the author of
													///< the last update of the TUS data currently registered on the server is identical with the specified NP ID.
													///< Processing is not performed when no TUS data is registered on the server. memset to 0 if comparison is not necessary.

			/// @brief
			/// The default constructor.
			///
			/// The default constructor.
			TusSetDataInputParams() : slotId(0), isVirtualUser(false), lastChangedDate(0)
			{
				memset(&requiredLastChangeUser, 0, sizeof(SceNpId));
			}
		}TusSetDataInputParams;

		/// @brief
		/// Represents input parameters used when getting TUS (title user storage) data.
		///
		/// Represents input parameters used when getting TUS (title user storage) data.
		typedef struct TusGetDataInputParams
		{
			int32_t		slotId;			///< The ID of the slot that the data belongs to.
			SceNpId		npid;			///< The NPID of the user to retrieve the data for.
			bool		isVirtualUser;	///< A flag that specifies whether this update is for a virtual user.
		}TusGetDataInputParams;

		/// @brief
		/// Represents a comment to censor or sanitize using the word filter.
		///
		/// Represents a comment to censor or sanitize using the word filter.
		typedef struct WordFilterParam
		{
			char comment[SCE_NP_SCORE_CENSOR_COMMENT_MAXLEN];///< The comment to be checked by the word filter. The maximum length of the comment is defined by #SCE_NP_SCORE_CENSOR_COMMENT_MAXLEN.
			int32_t resolveRetry;							///< UnSupported
			uint32_t resolveTimeout;						///< UnSupported
			uint32_t connTimeout;							///< The timeout for the communication. Only values of 10,000,000 or more are valid, and an error will occur if a smaller value is used.
			uint32_t sendTimeout;							///< UnSupported
			uint32_t recvTimeout;							///< UnSupported
			bool isAsync;									///< A flag that specifies whether to process the word filtering asynchronously or synchronously.
			char		padding[3];								
		}WordFilterParam;

		/// @brief
		/// Represents a comment that has been sanitized by the word filter.
		///
		/// Represents a comment that has been sanitized by the word filter.
		typedef struct WordFilterSanitized
		{
			char sanitizedComment[SCE_NP_SCORE_SANITIZE_COMMENT_MAXLEN];	///< The sanitized comment. The maximum length of the sanitized comment is defined by #SCE_NP_SCORE_SANITIZE_COMMENT_MAXLEN.
			size_t size;													///< The size of the sanitized comment.
		}WordFilterSanitized;

		}//end NP
	}//end Toolkit
}//end sce
#endif
